# Rust Integration Guide [Home](../../) > [Developers](../README.md) > Rust Integration guide to integrating with Noderr Protocol using Rust with production-ready code examples.

## HTTP Client Setup ```rust use reqwest::Client; use serde::{Deserialize, Serialize}; use std::env; #[derive(Serialize, Deserialize, Debug)] pub struct Vault { pub id: String, pub name: String, pub total_assets: f64, pub apy: f64, } #[derive(Serialize, Deserialize, Debug)] pub struct DepositRequest { pub amount: f64, } #[derive(Serialize, Deserialize, Debug)] pub struct DepositResponse { pub transaction_hash: String, pub status: String, } pub struct NoderrClient { client: Client, base_url: String, api_key: String, } impl NoderrClient { pub fn new(api_key: Option<String>) -> Self { let api_key = api_key.or_else(|| env::var("NODERR_API_KEY").ok()).expect("NODERR_API_KEY not found"); Self { client: Client::new(), base_url: "https://api.noderr.xyz/v1".to_string(), api_key, } } pub async fn get_vault(&self, vault_id: &str) -> Result<Vault, Box<dyn std::error::Error>> { let url = format!("{}/vaults/{}", self.base_url, vault_id); let response = self.client.get(&url).header("Authorization", format!("Bearer {}", self.api_key)).send().await?; Ok(response.json().await?) } pub async fn deposit(&self, vault_id: &str, amount: f64) -> Result<DepositResponse, Box<dyn std::error::Error>> { let url = format!("{}/vaults/{}/deposit", self.base_url, vault_id); let request = DepositRequest { amount }; let response = self.client.post(&url).header("Authorization", format!("Bearer {}", self.api_key)).json(&request).send().await?; Ok(response.json().await?) } pub async fn withdraw(&self, vault_id: &str, shares: f64) -> Result<DepositResponse, Box<dyn std::error::Error>> { let url = format!("{}/vaults/{}/withdraw", self.base_url, vault_id); let request = DepositRequest { amount: shares }; let response = self.client.post(&url).header("Authorization", format!("Bearer {}", self.api_key)).json(&request).send().await?; Ok(response.json().await?) } } ```

## GraphQL Client ```rust use graphql_client::{GraphQLQuery, Response}; use reqwest::Client; #[derive(GraphQLQuery)] #[graphql( schema_path = "schema.graphql", query_path = "queries/get_vault.graphql", response_derives = "Debug" )] pub struct GetVault; pub struct GraphQLClient { client: Client, endpoint: String, api_key: String, } impl GraphQLClient { pub fn new(endpoint: String, api_key: String) -> Self { Self { client: Client::new(), endpoint, api_key, } } pub async fn fetch_vault(&self, vault_id: &str) -> Result<Vault, Box<dyn std::error::Error>> { let variables = get_vault::Variables { id: vault_id.to_string(), }; let body = GetVault::build_query(variables); let response = self.client.post(&self.endpoint).header("Authorization", format!("Bearer {}", self.api_key)).json(&body).send().await?; let response_body: Response<get_vault::ResponseData> = response.json().await?; match response_body.data { Some(data) => Ok(data.vault), None => Err("No vault data returned".into()), } } } // Query definitions pub const GET_VAULT_QUERY: &str = r#" query GetVault($id: ID!) { vault(id: $id) { id name totalAssets apy strategies { id name allocation } } } "#; pub const GET_USER_VOTES_QUERY: &str = r#" query GetUserVotes($userId: ID!) { user(id: $userId) { id votes { proposalId vote votingPower } } } "#; pub const GET_VAULT_PERFORMANCE_QUERY: &str = r#" query GetVaultPerformance($vaultId: ID!, $period: String!) { vaultPerformance(vaultId: $vaultId, period: $period) { returns volatility sharpeRatio maxDrawdown } } "#; pub const GET_PROPOSALS_QUERY: &str = r#" query GetProposals($status: String!) { proposals(status: $status) { id title description votesFor votesAgainst endTime } } "#; ```

## WebSocket Subscriptions ```rust use tokio_tungstenite::{connect_async, tungstenite::Message}; use futures::{SinkExt, StreamExt}; use serde_json::json; pub async fn subscribe_to_prices(asset_id: &str) -> Result<(), Box<dyn std::error::Error>> { let (ws_stream, _) = connect_async("wss://api.noderr.xyz/ws").await?; let (mut write, mut read) = ws_stream.split(); let subscribe_msg = json!({ "type": "subscribe", "channel": format!("prices:{}", asset_id) }).to_string(); write.send(Message::Text(subscribe_msg)).await?; while let Some(msg) = read.next().await { match msg? { Message::Text(text) => { println!("Price update: {}", text); } Message::Close(_) => { println!("WebSocket closed"); break; } _ => {} } } Ok(()) } pub async fn subscribe_to_apy_changes(vault_id: &str) -> Result<(), Box<dyn std::error::Error>> { let (ws_stream, _) = connect_async("wss://api.noderr.xyz/ws").await?; let (mut write, mut read) = ws_stream.split(); let subscribe_msg = json!({ "type": "subscribe", "channel": format!("apy:{}", vault_id) }).to_string(); write.send(Message::Text(subscribe_msg)).await?; while let Some(msg) = read.next().await { match msg? { Message::Text(text) => { println!("APY update: {}", text); } _ => {} } } Ok(()) } pub async fn subscribe_to_governance() -> Result<(), Box<dyn std::error::Error>> { let (ws_stream, _) = connect_async("wss://api.noderr.xyz/ws").await?; let (mut write, mut read) = ws_stream.split(); let subscribe_msg = json!({ "type": "subscribe", "channel": "governance" }).to_string(); write.send(Message::Text(subscribe_msg)).await?; while let Some(msg) = read.next().await { match msg? { Message::Text(text) => { println!("Governance event: {}", text); } _ => {} } } Ok(()) } ```

## Smart Contract Interaction ```rust use ethers::prelude::*; use std::sync::Arc; #[derive(EthContract)] #[ethcontract( name = "Vault", raw_contract_json = "vault_abi.json" )] pub struct Vault; pub async fn get_vault_balance( vault_address: Address, user_address: Address, provider: Arc<Provider<Http>>, ) -> Result<U256, Box<dyn std::error::Error>> { let vault = Vault::new(vault_address, provider); let balance = vault.balance_of(user_address).call().await?; Ok(balance) } pub async fn deposit( vault_address: Address, amount: U256, wallet: LocalWallet, provider: Arc<Provider<Http>>, ) -> Result<TransactionReceipt, Box<dyn std::error::Error>> { let client = SignerMiddleware::new(provider, wallet); let vault = Vault::new(vault_address, Arc::new(client)); let tx = vault.deposit(amount).send().await?; let receipt = tx.await?; Ok(receipt.ok_or("Transaction failed")?) } pub async fn withdraw( vault_address: Address, shares: U256, wallet: LocalWallet, provider: Arc<Provider<Http>>, ) -> Result<TransactionReceipt, Box<dyn std::error::Error>> { let client = SignerMiddleware::new(provider, wallet); let vault = Vault::new(vault_address, Arc::new(client)); let tx = vault.withdraw(shares).send().await?; let receipt = tx.await?; Ok(receipt.ok_or("Transaction failed")?) } pub async fn monitor_transaction( tx_hash: H256, provider: Arc<Provider<Http>>, ) -> Result<TransactionReceipt, Box<dyn std::error::Error>> { let receipt = provider.get_transaction_receipt(tx_hash).await?.ok_or("Transaction not found")?; Ok(receipt) } ```

## Error Handling ```rust use thiserror::Error; #[derive(Error, Debug)] pub enum NoderrError { #[error("HTTP error: {0}")] HttpError(#[from] reqwest::Error), #[error("Serialization error: {0}")] SerializationError(#[from] serde_json::Error), #[error("Contract error: {0}")] ContractError(String), #[error("WebSocket error: {0}")] WebSocketError(String), #[error("Timeout error")] TimeoutError, #[error("Invalid input: {0}")] InvalidInput(String), } pub async fn safe_api_call<F, T>(f: F) -> Result<T, NoderrError> where F: std::future::Future<Output = Result<T, NoderrError>>, { f.await } pub async fn with_timeout<F, T>( f: F, duration: std::time::Duration, ) -> Result<T, NoderrError> where F: std::future::Future<Output = Result<T, NoderrError>>, { tokio::time::timeout(duration, f).await.map_err(|_| NoderrError::TimeoutError)? } pub async fn retry_with_backoff<F, T>( mut f: F, max_retries: u32, base_delay: std::time::Duration, ) -> Result<T, NoderrError> where F: FnMut() -> std::pin::Pin<Box<dyn std::future::Future<Output = Result<T, NoderrError>>>>, { let mut delay = base_delay; for attempt in 0..max_retries { match f().await { Ok(result) => return Ok(result), Err(e) => { if attempt == max_retries - 1 { return Err(e); } tokio::time::sleep(delay).await; delay = delay.mul_f32(2.0); } } } Err(NoderrError::ContractError("Max retries exceeded".to_string())) } ```

## Data Validation ```rust use validator::Validate; #[derive(Validate, Serialize, Deserialize)] pub struct DepositInput { #[validate(length(min = 1, max = 100))] pub vault_id: String, #[validate(range(min = 0.0))] pub amount: f64, } #[derive(Validate, Serialize, Deserialize)] pub struct WithdrawInput { #[validate(length(min = 1, max = 100))] pub vault_id: String, #[validate(range(min = 0.0))] pub shares: f64, } pub fn validate_ethereum_address(address: &str) -> Result<Address, NoderrError> { address.parse::<Address>().map_err(|_| NoderrError::InvalidInput("Invalid Ethereum address".to_string())) } pub fn validate_amount(amount: f64) -> Result<f64, NoderrError> { if amount <= 0.0 { return Err(NoderrError::InvalidInput("Amount must be positive".to_string())); } Ok(amount) } ```

## Async Runtime ```rust #[tokio::] async fn () -> Result<(), Box<dyn std::error::Error>> { // Initialize client let client = NoderrClient::new(None); // Fetch vault let vault = client.get_vault("vault-1").await?; println!("Vault: {:?}", vault); // Deposit let deposit_result = client.deposit("vault-1", 100.0).await?; println!("Deposit result: {:?}", deposit_result); // Subscribe to prices tokio::spawn(async { if let Err(e) = subscribe_to_prices("ETH").await { eprintln!("WebSocket error: {}", e); } }); // Keep running tokio::signal::ctrl_c().await?; Ok(()) } ```

## Best Practices Summary 1. **Use async/await** for concurrent operations 2. **Implement proper error handling** with thiserror 3. **Use type-safe contract bindings** with ethers-rs 4. **Validate inputs** at compile time with validator 5. **Use Tokio** for async runtime 6. **Implement connection pooling** with reqwest 7. **Log errors** appropriately 8. **Use strong typing** throughout --- **See Also:** [REST API Documentation](../api-documentation/rest-api.md) | [GraphQL API Documentation](../api-documentation/graphql-api.md) | [Smart Contract Integration](../smart-contract-integration.md) --- **See Also:** - [Noderr White Paper v7.1](../../white-papers/Noderr_White_Paper_v7.1.md) - [Noderr Lite Paper v3.1](../../white-papers/Noderr_Lite_Paper_v3.1.md) - [Noderr Protocol Documentation](README.md)
